/*--------------------- revision history -----------------------------------
 * (C) JA Doornik
 *
 * 14- 1-96 (JAD): replaced time and date functions 
 *  1- 3-93 (JAD): Created out of dldataio.c for third party interface
 *                 to .IN7/.BN7 files.
 *--------------------------------------------------------------------------*/

#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "in7bn7io.h"

#define MAX_PROJECTS 10
#define MAX_LINE_LEN 100

/*======================= external variables ===============================*/
double g_dMisval = -9999.99;                               /* missing value */
struct dbType g_db = { ABS_MX_VAR, 0, 0 };                 /* the data base */

/* note that variables are stored in
       g_db.mXt[0][0..g_db.cT-1]..g_db.mXt[g_db.cX-1][0..g_db.cT-1]
   marked for memory consistency checking is:
       g_db.mXt[*][g_db.cT]
   allocated is
       g_db.mXt[0][0..g_db.cT-1]..g_db.mXt[g_db.cX-1][0..g_db.cT-1]
   and
       g_db.mXt[g_db.cX]..g_db.mXt[g_db.mxX - 1]
*/


static char sErrOpenFmt[] = "Cannot open %s";
static char *sUnexpectedEof = "Unexpected end-of-file";
static char sChart[100];                   /* temporary buffer for messages */

static char *asKeywords[] =                    /* keywords at start of line */
{   "pcgive", "data", ";", ">", "-", "\n", "end",
};
static int aiKeylengths[] =                     /* length of those keywords */
{   6, 4, 1, 1, 1, 1, 3
};
enum iKeywords
{   INF_START, INF_DATA, INF_COMMENT, INF_VAR, INF_DEL, INF_EMPTY, INF_END
};

/*========================= general services ===============================*/

#define MyMalloc  malloc            /* standard memory allocation functions */
#define MyCalloc  calloc
#define MyRealloc realloc
#define VecAlloc(dim)  ((VECTOR)MyMalloc( (dim) * sizeof(double)))

void PopErrorMessage(char *sMsg)
{
    fprintf(stderr, "\nin7bn7io error: %s\n", sMsg);
}
void PopWarningMessage(char *sMsg)
{
    fprintf(stderr, "\nin7bn7io warning: %s\n", sMsg);
}

/*------------------------- SFileBase/SFileExt -----------------------------
 *  []Summary
 *    #include "in7bn7io.h"
 *
 *    char * SFileBase(char *sFilename)
 *    char * SFileExt(char *sFilename)
 *      sFilename     file name
 *
 *  []Return value
 *    SFileBase returns pointer to start of base of name.
 *    SFileExt  returns returns pointer to start of extension (the dot),
 *          or the ending \0 if there is no extension.
 *
 *  []Description
 *    A MS-DOS file name is: path + base + extension (incl. dot).
 *    These function locate the base and the extension.
 *--------------------------------------------------------------------------*/
char *SFileBase(char *sFilename)
{
    char *s;

    if ( (s = strrchr(sFilename, '\\')) != NULL)
        s++;                      /* points to first character after last \ */
    else if ( (s = strrchr(sFilename, ':')) != NULL)
        s++;                      /* points to first character after last : */
    else
        s = sFilename;                             /* whole name, no : or \ */
return(s);
}
char *SFileExt(char *sFilename)
{
    char *s;

    if ( (s = strrchr(sFilename, '.')) == NULL)
        s = sFilename + strlen(sFilename);    /* points to extension or end */
return s;
}
/*------------------------- END SFileBase/SFileExt -------------------------*/

/*----------------------------- FCheckCloseW -------------------------------
 *  []Summary
 *    #include "in7bn7io.h"
 *
 *    bool FCheckCloseW(FILE *fo)
 *      fo      in: output file for checking.
 *
 *  []Return value
 *    Returns TRUE for no error,FALSE for error.
 *
 *  []Description
 *    FCheckCloseW checks for write error on the file fo,
 *    pops up error message if error occurred and closes the file.
 *
 *    NB. error message 2 (No such file or directory) and 17 (File exists)
 *    are not really error messages: 2 is issued when opening a new file,
 *    17 when opening an existing file.
 *    NB. This is compiler and MS-DOS specific!!!!!!!!
 *--------------------------------------------------------------------------*/
bool FCheckCloseW(FILE *fo)
{
    bool ret_val = TRUE;

    if (ferror(fo) && errno == ENOSPC)
    {
        PopErrorMessage("Disk Full!");
        ret_val = FALSE;
    }
    fclose(fo);

return ret_val;
}
/*----------------------------- END FCheckCloseW ---------------------------*/


/*======================== time/date functions =============================*/
static char sTime[16], sDate[16];
/*---------------------------- SDate/STime ---------------------------------
 *  []Summary
 *    #include "in7bn7io.h"
 *
 *    char *SDate(void)
 *    char *STime(void)
 *
 *   []Return value
 *     SDate returns a pointer to the date in day-month-year format
 *     (eg " 1-02-1992").
 *     STime returns a string containing the current system time.
 *     The format is "hh:mm:ss.99".
 *
 *     Both SDate and STime use the same single statically allocated
 *     buffer for holding the return string. Each call to one of these
 *     routines destroys the result of the previous call.
 *--------------------------------------------------------------------------*/
char *SDate(void)
{
    struct tm *atime;
	time_t aclock;
	
	time(&aclock);                 				     /* Get time in seconds */
	atime = localtime(&aclock);  				  /* Convert time to struct */
					 
    sprintf(sDate, "%2u-%02u-%4u",
        atime->tm_mday, 1 + atime->tm_mon, 1900 + atime->tm_year);
return sDate;
}
char *STime(void)
{
    struct tm *atime;
	time_t aclock;
	
	time(&aclock);                 				     /* Get time in seconds */
	atime = localtime(&aclock);  				  /* Convert time to struct */

    sprintf(sTime, "%2d:%02d:%02d", atime->tm_hour, atime->tm_min, atime->tm_sec);
	
return sTime;
}
/*-------------------------- END SDate/STime -------------------------------*/

/*============================= database ===================================*/

/*----------------------- DbKill/DbSetcT/DbSetSam --------------------------
 *  []Summary
 *    #include "in7bn7io.h"
 *
 *    int  DbSetcT(int cT)
 *    void DbSetSam(int iFreq, SAMPLE sam)
 *    void DbKill(void)
 *
 *  []Return value
 *    DbSetcT returns the installed no of observations: min(cT, ABS_MX_OBS).
 *    The rest has no return value.
 *
 *  []Description
 *    DbSetcT sets the number of observations.
 *    DbSetSam sets the data base frequency and sample, sets reserved vars.
 *    DbKill frees the whole data base
 *--------------------------------------------------------------------------*/
int  DbSetcT(int cT)
{
    g_db.cT = min(cT, ABS_MX_OBS);

return g_db.cT;
}
void DbSetSam(int iFreq, SAMPLE sam)
{
    g_db.iFreq = iFreq;
    g_db.sam   = sam;
}
void DbKill(void)
{
    int i;

    for (i = g_db.cX - 1; i >= 0; i--)                        /* deallocate */
    {
        free(g_db.mXt[i]);
        free(g_db.asX[i]);
        free(g_db.asComment[i]);
    }
    for (i = 0; i < ABS_MX_VAR; i++)                         /* set to zero */
        g_db.mXt[i] = NULL, g_db.asX[i] = g_db.asComment[i] = NULL;

    free(g_db.sComment);  g_db.sComment = NULL;
    g_db.cX = g_db.cT = g_db.iFreq = 0;
    g_db.sam.year1 = g_db.sam.year2 = 0;
    g_db.fChange = FALSE;                  /* to indicate unchanged dataset */
}
/*------------------------ END DbKill/DbSetcT/DbSetSam ---------------------*/

/*-------------------------- IDbAdd/IDbDel ---------------------------------
 *  []Summary
 *    #include "in7bn7io.h"
 *
 *    int  IDbAdd(int cAdd)
 *    int  IDbDel(int iVar)
 *
 *  []Return value
 *    IDbAdd()  Returns -1 if couldn't add cAdd vars, returns index of
 *    first created variable otherwise.
 *    IDbDel()  returns -1 if argument was not a variable index, else
 *    returns new highest variable index.
 *
 *  []Description
 *    IDbAdd()  add cAdd variables to the database. Returns -1 if couldn't
 *        add cAdd vars, returns index of first created variable otherwise.
 *    IDbDel()  deletes specified variable
 *--------------------------------------------------------------------------*/
int  IDbAdd(int cAdd)
{
    int  i, j;

    if (g_db.cX + cAdd >= g_db.mxX)
    {
        PopErrorMessage("Maximum number of variables reached in database");
        return -1;
    }

    for (i = g_db.cX; i < g_db.cX + cAdd; i++)
    {   g_db.mXt[i] = VecAlloc(g_db.cT + 1);
        g_db.asX[i] = (char *)MyMalloc((NAME_LEN + 4) * sizeof(char));
    }
                                       /* now check last allocated variable */
    if (g_db.mXt[--i] == NULL || g_db.asX[i] == NULL)   /* return if failed */
    {
        for (i = g_db.cX; i < g_db.cX + cAdd; i++)
        {   free(g_db.mXt[i]);
            free(g_db.asX[i]);
        }
        sprintf(sChart, "Cannot add %d variable(s) to database", cAdd);
        PopErrorMessage(sChart);
        return -1;
    }
    for (i = 0; i < cAdd; i++)
    {    /* mark first and last element, enables detection of memory errors */
        g_db.mXt[g_db.cX][g_db.cT] = DATAMARK;
        for (j = 0; j < g_db.cT; j++)
            g_db.mXt[g_db.cX][j] = g_dMisval;
        sprintf(g_db.asX[g_db.cX], "var%d", g_db.cX + 1);

        g_db.cX++;                                   /* succesfully created */
    }

return g_db.cX - cAdd;                 /* index first added var in g_db.mXt */
}
int  IDbDel(int iVar)
{
    int  i;

    if (iVar < 0 || iVar >= g_db.cX)
        return -1;

    g_db.mXt[iVar][g_db.cT] = 0.;                                 /* unmark */
    free(g_db.mXt[iVar]);
    free(g_db.asX[iVar]);

    for (i = iVar; i < g_db.cX - 1; i++)
    {
        g_db.mXt[i]  = g_db.mXt[i + 1];
        g_db.asX[i]  = g_db.asX[i + 1];
    }

    g_db.mXt[g_db.cX] = NULL;  g_db.asX[g_db.cX] = NULL;
    g_db.cX--;

return g_db.cX - 1;
}
/*------------------------- END IDbAdd/IDbDel ------------------------------*/

/*----------------- IObsYear/IObsPeriod/IDateObs/IGetcT --------------------
 *  []Summary
 *    #include "in7bn7io.h"
 *
 *    int  IDateObs(int iyear, int iperiod)
 *    int  IGetcT(SAMPLE *sam, int iFreq)
 *    int  IObsPeriod(int iObs)
 *    int  IObsYear(int iObs)
 *
 *  []Return value
 *    IDateObs   returns the observation number of the date in the database.
 *    IGetcT returns the # of observations in the sample,
 *    given the specified frequency.
 *    IObsPeriod returns the period of the observation in the database.
 *    IObsYear   returns the year of the observation in the database.
 *--------------------------------------------------------------------------*/
int  IObsYear(int iObs)
{
    int nyr = iObs / g_db.iFreq;
    int nper = g_db.sam.period1 + iObs - nyr * g_db.iFreq;
    if (nper > g_db.iFreq)  ++nyr;

return g_db.sam.year1 + nyr;
}
int  IObsPeriod(int iObs)
{
    int nyr = iObs / g_db.iFreq;
    int nper = g_db.sam.period1 + iObs - nyr * g_db.iFreq;
    if (nper > g_db.iFreq)  nper -= g_db.iFreq;

return nper;
}
int  IDateObs(int iyear, int iperiod)
{
    return((iyear - g_db.sam.year1) * g_db.iFreq - g_db.sam.period1 + iperiod);
}
int  IGetcT(SAMPLE *sam, int iFreq)
{
return((sam->year2 - sam->year1) * iFreq + sam->period2 - sam->period1 + 1);
}
/*----------------- END IObsYear/IObsPeriod/IDateObs/IGetcT ----------------*/


/*------------------ SetSample/SetSampleEnd/SetUnionSample -----------------
 *  []Summary
 *    #include "in7bn7io.h"
 *
 *    void SetSampleEnd(SAMPLE *sam, int iFreq, int cT)
 *    void SetSample(int iT1, int iT2, SAMPLE *sam)
 *    void SetUnionSample(SAMPLE *sam2, SAMPLE *sam1, int iFreq)
 *
 *  []No return value
 *
 *  []Description
 *    SetSampleEnd: given sam.year1 and sam.period1, works out sam.year2
 *      and sam.period2 from iFreq and cT.
 *    SetSample: sets the sample period for observations iT1..iT2 in the
 *      database.
 *    SetUnionSample: changes sam2, so that the new sam2 is the union of
 *      sam1 and sam2.
 *--------------------------------------------------------------------------*/
void SetSampleEnd(SAMPLE *sam, int iFreq, int cT)
{
    int  nyr = --cT / iFreq;                   /* translate cT to obs index */
    int  nper = sam->period1 + cT - nyr * iFreq;
    if (nper > iFreq)  ++nyr, nper -= iFreq;

    sam->year2 = sam->year1 + nyr;
    sam->period2 = nper;
}
void SetSample(int iT1, int iT2, SAMPLE *sam)
{
    sam->year1 = IObsYear(iT1);
    sam->period1 = IObsPeriod(iT1);
    sam->year2 = IObsYear(iT2);
    sam->period2 = IObsPeriod(iT2);
}
void SetUnionSample(SAMPLE *sam2, SAMPLE *sam1, int iFreq)
{
    if (sam2->year1 * iFreq + sam2->period1 > sam2->year2 * iFreq + sam2->period2)
    {   *sam2 = *sam1;
        return;
    }

    if (sam2->year1 > sam1->year1)
        sam2->year1 = sam1->year1, sam2->period1 = sam1->period1;
    else if (sam2->year1 == sam1->year1)
        sam2->period1 = min(sam2->period1, sam1->period1);

    if (sam2->year2 < sam1->year2)
        sam2->year2 = sam1->year2, sam2->period2 = sam1->period2;
    else if (sam2->year2 == sam1->year2)
        sam2->period2 =  max(sam2->period2, sam1->period2);

    if (sam2->period1 > iFreq)    sam2->period1 = iFreq;
    if (sam2->period1 < 1)        sam2->period1 = 1;
    if (sam2->period2 > iFreq)    sam2->period2 = iFreq;
    if (sam2->period2 < 1)        sam2->period2 = 1;
}
/*---------------- END SetSample/SetSampleEnd/SetUnionSample ---------------*/

/*---------------------------- SNameToken ----------------------------------
 *  []Summary
 *    #include "in7bn7io.h"
 *
 *    char *SNameToken(char *sName)
 *
 *  []Return value
 *    Returns sName
 *
 *  []Description
 *    Naming conventions: a name constists of upto NAME_LEN chars with an
 *    optional
 *    lag field of three positions appended, eg:
 *          cons_1, variable, variable_10.
 *
 *    SNameToken: converts a variable name into a pcgive name, by checking
 *      length, removing trailing and leading spaces. Note that the
 *      resulting variable name is NOT necessarily a proper token in
 *      Algebra sense!
 *--------------------------------------------------------------------------*/
char *SNameToken(char *sName)
{
    int   i, j, len = strlen(sName);

    if (len > NAME_LEN)                             /* shorten if necessary */
        sName[len = NAME_LEN] = '\0';

    for (i = 0; sName[i] == ' '; i++ )  ;           /* count leading spaces */
    for (j = 0; i <= len; i++, j++)  sName[j] = sName[i];    /* remove them */

    for (i = strlen(sName) - 1; i > 0 && sName[i] <= ' '; i--) ;
    sName[i + 1] = '\0';                  /* delete trailing chars <= space */

    for (i = 0; i < (int)strlen(sName); i++)
        if (sName[i] == '\"' || sName[i] == ' ')
            sName[i] = '_';           /* replace double quote by underscore */

return(sName);
}
/*--------------------------- END SNameToken -------------------------------*/


/*======================= input/output support =============================*/

/*-------------------------- FReadBinary/lSaveBinary ------------------------
 *  []Summary
 *    #include "in7bn7io.h"
 *
 *    int FReadBinary(FILE *fi, long lAddr, VECTOR vX, int iT1, int iT2,
 *        int cT, char *sX)
 *       fi        in:  pointer to file to read from
 *       lAddr     in:  address in file of variable
 *       vXt       in:  destination data vector
 *                 out: mXt[iX1..iX1+cX-1][iT1..iT1+cT-1] holds read data
 *       iT1,iT2   in:  sample period of variable to read, expressed
 *                      relative to database
 *       cT        in:  # of observations to read from file
 *       sX        in:  name of variable to read
 *
 *    long LSaveBinary(FILE *fo, VECTOR vX, int cT, int iYear1, int iPeriod1,
 *        int iFreq, char *sX)
 *       fo        in:  pointer to file to write to
 *       vXt       in:  data vector to write
 *       cT        in:  # of observations to write to file
 *       iYear1    in:  year of first observation in vX
 *       iPeriod1  in:  period of first observation in vX
 *       iFreq     in:  data frequency of vX
 *       sX        in:  name of variable to read
 *
 *  []Return value
 *    FReadBinary returns TRUE if successful..
 *    LSaveBinary returns the address of start of actual variable
 *
 *  []Description
 *    FReadBinary reads a variable from a .BN7 file.
 *    LSaveBinary saves a data vector to a .BN7 file.
 *--------------------------------------------------------------------------*/
int FReadBinary(FILE *fi, long lAddr, VECTOR vX, int iT1, int iT2,
    int cT, char *sX)
{
    unsigned tread;

    if (iT1 >= cT || iT2 <= 0)
    {
        sprintf(sChart, "%s has no overlap with database sample", sX);
        PopErrorMessage(sChart);
        return FALSE;                                         /* no overlap */
    }

    if (iT1 < 0)
        lAddr += abs(iT1) * sizeof(double), iT1 = 0;
    if (iT2 >= cT)
        iT2 = cT - 1;

    if (fseek(fi, lAddr, SEEK_SET) != 0
        || (tread = fread(&vX[iT1], sizeof(double), iT2 - iT1 + 1, fi)) == 0)
    {
        sprintf(sChart, "Start of %s not found", sX);
        PopErrorMessage(sChart);
        return FALSE;
    }

    if (tread != (unsigned)iT2 - iT1 + 1)
    {
        sprintf(sChart, "%s while reading %s", sUnexpectedEof, sX);
        PopWarningMessage(sChart);
    }

return TRUE;
}
static void setSignature(struct SigType *sig, int cT, int iYear1, int iPeriod1,
    int iFreq, char *sX)
{
    memset(sig, '\0', sizeof(struct SigType));
    sig->dSig = DATAMARK;
    sig->cT = (short int)cT;
    strncpy(sig->sX, sX, min(12, strlen(sX)));
    sig->iPeriod1 = (short int)iPeriod1;
    sig->iYear1 = (short int)iYear1;
    sig->iFreq = (short int)iFreq;
}
long LSaveBinary(FILE *fo, VECTOR vX, int cT, int iYear1, int iPeriod1,
    int iFreq, char *sX)
{
    struct SigType signature;  long l;

    if (cT <= 0)
        return -1L;                                      /* nothing to save */

    setSignature(&signature, cT, iYear1, iPeriod1, iFreq, sX);

    fwrite(&signature, sizeof(struct SigType), 1, fo);   /* write signature */
    l = ftell(fo);                   /* address of start of actual variable */
    fwrite(vX, sizeof(double), cT, fo);                   /* write variable */

return l;
}
/*------------------------ END FReadBinary/LSaveBinary ---------------------*/

/*------------------ InfAlloc/InfFree/IdxAppend ----------------------------
 *  []Summary
 *    #include "in7bn7io.h"
 *
 *    struct InfType *InfAlloc(char *sFilename)
 *        sFilename       in: filename to which inf corresponds.
 *    void InfFree(struct InfType *inf)
 *        inf             in:  structure to free
 *                        out: all contents freed
 *    struct IdxType *IdxAppend(struct InfType *inf, char *sName)
 *        inf             in:  structure to append to
 *        sName           in:  name of variable to append
 *
 *  []Return value
 *    InfAlloc returns NULL if failed.
 *    IdxAppend returns pointer to appended structure.
 *
 *  []Description
 *    InfAlloc allocates and resets a InfType structure, InfFree frees it.
 *    IdxAppend appends a variable.
 *--------------------------------------------------------------------------*/
struct InfType *InfAlloc(char *sFilename)
{
    struct InfType *inf;

    if ( (inf = MyCalloc(1, sizeof(struct InfType))) == NULL)
        return(NULL);

    inf->cX = inf->cT = 0;
    inf->iFreq = 1;  inf->iType = RS_BINARY_VAR;

    if (sFilename) strcpy(inf->sFilename, sFilename);

    inf->asProject = (char **)MyCalloc(MAX_PROJECTS, sizeof(char *));
    inf->iProject = -1;                                       /* no project */

return(inf);
}
void InfFree(struct InfType *inf)
{
    int  i;  struct IdxType *pdx1, *pdx2;

    if (inf == NULL)
        return;

    if (inf->asProject)
    {
        for (i = 0; i < MAX_PROJECTS; i++)
            free(inf->asProject[i]);

        free(inf->asProject);
    }
    free(inf->sComment);

    for (pdx1 = inf->pIdx; pdx1 != NULL; pdx1 = pdx2)
    {   
        pdx2 = pdx1->next;
        free(pdx1->sX);
        free(pdx1->sComment);
        free(pdx1);
    }

    free(inf);
}
struct IdxType *IdxAppend(struct InfType *inf, char *sName)
{
    struct IdxType *pdx1, *pdx2, *pdxend;

    if ( (pdx2 = MyCalloc(1, sizeof(struct IdxType))) == NULL)
        return NULL;

    pdx2->sX = strdup(sName);

    pdxend = NULL;                                      /* find end of list */
    for (pdx1 = pdxend = inf->pIdx; pdx1 != NULL; pdxend = pdx1, pdx1 = pdx1->next)
    {
        if (strcmp(pdx1->sX, sName) == 0)
        {
            pdx1->flag |= BIN_OLDGENERATION;
            pdx2->iGeneration = pdx1->iGeneration + 1;
        }
    }
    if (pdxend == NULL)
        inf->pIdx = pdx2;                               /* is start of list */
    else
        pdxend->next = pdx2;

return pdx2;
}
/*------------------ END InfAlloc/InfFree/IdxAppend ------------------------*/

/*-------------------- FSetProject/SelectProject ---------------------------
 *  []Summary
 *    #include "in7bn7io.h"
 *
 *    bool FSetProject(char *sProject, struct InfType *inf, struct IdxType *idx)
 *        sProject    in: project name
 *        inf         in: list of variables
 *        idx         in: variable info
 *
 *    void SelectProject(struct InfType *inf, int iProject)
 *        inf         in: whole file info
 *        iProject    in: project nr. to select (-1: select all projects)
 *
 *  []Return value
 *    FSetProject returns TRUE if project successfully appended.
 *
 *  []Description
 *    FSetProject adds a project to the list.
 *    SelectProject selects a project for reading.
 *--------------------------------------------------------------------------*/
bool FSetProject(char *sProject, struct InfType *inf, struct IdxType *idx)
{
    int  i;

    for (i = 0; i < inf->cProject; i++)
    {
        if (strcmp(inf->asProject[i], sProject) == 0)   /* already defined */
        {   idx->iProject = i;
            return TRUE;
        }
    }
    if (inf->cProject < MAX_PROJECTS)
        inf->asProject[inf->cProject++] = strdup(sProject);
    else
        return FALSE;

    idx->iProject = inf->cProject - 1;

return TRUE;
}
void SelectProject(struct InfType *inf, int iProject)
{
    struct IdxType *pdx;

    inf->cX = 0;

    for (pdx = inf->pIdx; pdx != NULL; pdx = pdx->next)
    {
        if (pdx->flag & BIN_OLDGENERATION)
            continue;             /* don't select deleted and old variables */

        if (iProject == -1 || iProject == pdx->iProject)          /* select */
        {
            pdx->flag |= BIN_SELECTED;
            inf->cX++;
        }
    }
}
/*-------------------- END FSetProject/SelectProject -----------------------*/

/*---------------------- DbToInf/InfToDb/InfSample -------------------------
 *  []Summary
 *    #include "in7bn7io.h"
 *
 *    void DbToInf(struct InfType *inf)
 *    void InfSample(struct InfType *inf)
 *    void InfToDb(struct InfType *inf)
 *        inf     in: .IN7 information
 *
 *  []No return value
 *
 *  []Description
 *    DbToInf copies database information to inf.
 *    InfToDb copies information in inf to the database.
 *    InfSample determines the common sample in the inf selection.
 *
 *--------------------------------------------------------------------------*/
void DbToInf(struct InfType *inf)
{
    int  i, t1, t2;  struct IdxType *pdx;

    inf->cT = g_db.cT;
    inf->iFreq = inf->iFreqDb = g_db.iFreq;
    inf->sam = inf->samDb = g_db.sam;

    if (g_db.sComment)
        inf->sComment = strdup(g_db.sComment);

    for (i = inf->cX = 0; i < g_db.cX; i++)
    {
        if ( (pdx = IdxAppend(inf, g_db.asX[i])) != NULL)
        {
            pdx->flag |= BIN_SELECTED;
            pdx->sam = g_db.sam, pdx->iFreq = g_db.iFreq, ++inf->cX;
            if (g_db.asComment[i])
                pdx->sComment = strdup(g_db.asComment[i]);

             /* don't save leading/trailing missing values, t1..t2 is saved */
            for (t1 = 0; t1 < g_db.cT && g_db.mXt[i][t1] == g_dMisval; t1++)
                ;
            for (t2 = g_db.cT - 1; t2 > 0 && g_db.mXt[i][t2] == g_dMisval; t2--)
                ;

            SetSample(t1, t2, &pdx->sam);
        }
    }
}
void InfSample(struct InfType *inf)
{
    struct IdxType *pdx;
    SAMPLE sam;  int freq;

    sam.year1 = sam.year2 = sam.period1 = sam.period2 = 0;

    for (pdx = inf->pIdx, freq = 0; pdx != NULL; pdx = pdx->next)
    {
        if (pdx->flag & BIN_SELECTED)
        {
            if (freq == 0)
            {
                freq = pdx->iFreq;
                sam = pdx->sam;
            }
            else if (freq != pdx->iFreq)
            {
                sprintf(sChart, "Frequency inconsistency, %s dropped", pdx->sX);
                PopWarningMessage(sChart);
                pdx->flag &= ~BIN_SELECTED;
                inf->cX--;
                continue;
            }
            SetUnionSample(&sam, &pdx->sam, freq);
        }
    }
    inf->iFreq = freq;  inf->sam = sam;
    inf->cT = IGetcT(&sam, freq);
}
void InfToDb(struct InfType *inf)
{
    if (inf->sComment)
        g_db.sComment = strdup(inf->sComment);

             /* copying of names is done when variable is read successfully */

    inf->cT = DbSetcT(inf->cT);
    SetSampleEnd(&inf->sam, inf->iFreq, inf->cT);/*truncate if too many obs */
    DbSetSam(inf->iFreq, inf->sam);
}
/*--------------------- END DbToInf/InfToDb/InfSample ----------------------*/

/*-------------------- iGetKeyline/iNextKeyline ----------------------------
 *    fi        in:  pointer to input file
 *    sLine     in:  line buffer
 *              out: holds next line from file
 *
 * iGetKeyline reads a line from the file, and returns the identifier which
 *   starts the line.
 * iNextKeyline reads the next line from the file, skipping any comment lines.
 *   Returns the identifier which starts the line.
 * sGetComment returns a pointer to a buffer holding comment.
 *--------------------------------------------------------------------------*/
static enum iKeywords iGetKeyline(FILE *fi, char *sLine)
{
    int i;

    if (fgets(sLine, MAX_LINE_LEN, fi) == NULL)                /* read line */
        return EOF;
    else
        for (i = INF_START; i <= INF_END; i++)
            if (strnicmp(sLine, asKeywords[i], aiKeylengths[i]) == 0)
                return i;

    PopWarningMessage("Unexpected line found");
return EOF;
}
static enum iKeywords iNextKeyline(FILE *fi, char *sLine)
{
    int i;

    while ( (i = iGetKeyline(fi, sLine)) != EOF
          && (i == INF_COMMENT || i == INF_EMPTY) )
        ;

return i;
}
static char *sGetComment(FILE *fi, char *sLine)
{
    int i, ch, len = 0;  char *scomment = NULL;

    while ( (ch = fgetc(fi)) == *asKeywords[INF_COMMENT])
    {
        ungetc(ch, fi);
        if ( (i = iGetKeyline(fi, sLine)) == EOF)
            return scomment;

        scomment = MyRealloc(scomment, len + strlen(sLine) * sizeof(char));
        if (scomment)
        {
            strcpy(scomment + len, sLine + 1);
            len = strlen(scomment);
        }
    }
    ungetc(ch, fi);

return scomment;
}
/*-------------------- END iGetKeyline/iNextKeyline ------------------------*/

/*------------- FReadInfFile/writeComment/FSaveInfFile ---------------------
 *  []Summary
 *    #include "in7bn7io.h"
 *
 *    bool FReadInfFile(struct InfType *inf, bool fReadVars, bool fDoDel)
 *          inf->sFilename  in: .IN7 file name
 *          inf->sDatafile  in: .BN7 file name:
 *                               extension of .IN7 file + base name recorded on
 *                               .IN7 file
 *          inf->sComment   out: data file comment
 *          inf->cX         out: # of variables in file
 *          inf->aIdx[cSel] out: set for each variable
 *          fReadVars       in:  TRUE: reads description of variables
 *          fDoDel          in:  TRUE: include deleted variables
 *
 *    bool FSaveInfFile(struct InfType *inf, bool fAppend)
 *       inf       in: information structure
 *       fAppend   in: TRUE: append to file
 *
 *  []Return value
 *    FReadInfFile returns TRUE if description read successfully,
 *    FALSE otherwise.
 *    FSaveInfFile returns TRUE if description written successfully,
 *    FALSE otherwise.

 *  []Description
 *    FSaveInfFile writes the data description block.
 *    FReadInfFile reads the data description block.
 *--------------------------------------------------------------------------*/
bool FReadInfFile(struct InfType *inf, bool fReadVars, bool fDoDel)
{
    int  key;  bool ret_val = FALSE;
    char line[MAX_LINE_LEN], *s1, *s2, sProject[20];
    FILE *fi;
    struct IdxType *pdx;

    if (!*SFileExt(inf->sFilename))
        strcat(inf->sFilename, ".IN7");

    if ( (fi = fopen(inf->sFilename, "r")) == NULL)/* try to open .IN7 file */
    {
        sprintf(sChart, sErrOpenFmt, inf->sFilename);
        PopErrorMessage(sChart);
        return FALSE;
    }

    strcpy(inf->sDatafile, inf->sFilename);
    strcpy(SFileExt(inf->sDatafile), ".BN7");      /* default bin file name */

    if (iGetKeyline(fi, line) != INF_START)                   /* identifier */
    {
        PopErrorMessage("This is not a PcGive .IN7 file");
        goto LReturn;
    }
    if (iNextKeyline(fi, line) != INF_DATA)             /* data description */
    {
        PopErrorMessage("Data description not found");
        goto LReturn;
    }
    sscanf(line + aiKeylengths[INF_DATA], " %s", SFileBase(inf->sDatafile));

    inf->sComment = sGetComment(fi, line);

    if (fReadVars)
    {
        for (;;)
        {
            do
            {   if ( (key = iNextKeyline(fi, line)) == EOF)
                {   ret_val = TRUE;
                    goto LReturn;
                }
            } while (key != INF_VAR && key != INF_DEL);

            if (key == INF_DEL && !fDoDel)
                continue;           /* deleted variables were not requested */

            s1 = line + 1;  pdx = NULL;
            if ( (s2 = strchr(s1, ' ')) != NULL)
            {   *s2 = '\0';                         /* end of variable name */
                pdx = IdxAppend(inf, SNameToken(s1));
            }
            if (pdx)
            {   sscanf(s2 + 1, " %d %d %d %d %d %ld %s %s %s ",
                    &pdx->sam.year1, &pdx->sam.period1,
                    &pdx->sam.year2, &pdx->sam.period2, &pdx->iFreq,
                    &pdx->lAddress, &sProject, &pdx->sDate, &pdx->sTime);
                FSetProject(sProject, inf, pdx);
                pdx->sComment = sGetComment(fi, line);
                if (key == INF_DEL)
                    pdx->flag |= BIN_DELETED;
                inf->cX++;
            }
        }
    }
    ret_val = TRUE;

LReturn:

    fclose(fi);

    return ret_val;
}

static void writeComment(FILE *fo, char *sComment)
{
    int i, ch;

    if (sComment && sComment[0])
    {
        for (i = ch = 0; i < (int)strlen(sComment); i++)
        {
            ch = sComment[i];
            if (i == 0 || sComment[i - 1] == '\n')
                if (ch != ';')  fputc(';', fo);
            fputc(ch, fo);
        }
        if (ch != '\n')  fputc('\n', fo);
    }
}
bool FSaveInfFile(struct InfType *inf, bool fAppend)
{
    FILE *fo;  char *sdate, *stime, *sp, sdefproject[13];  int i;
    struct IdxType *pdx;

    if (!*SFileExt(inf->sFilename))
        strcat(inf->sFilename, ".IN7");

    if ( (fo = fopen(inf->sFilename, (fAppend) ? "a" : "w")) == NULL)
    {
        sprintf(sChart, sErrOpenFmt, inf->sFilename);
        PopErrorMessage(sChart);
        return(FALSE);
    }

    strcpy(sdefproject, SFileBase(inf->sFilename)); /* base name as default */
    *(SFileExt(sdefproject)) = '\0';                        /* project name */

    if (!fAppend)
    {
        fprintf(fo, "%s %d\n", asKeywords[INF_START], VERSION_NR);  /* mark */

        fprintf(fo, "%s %s\n", asKeywords[INF_DATA], SFileBase(inf->sDatafile));

        writeComment(fo, inf->sComment);                    /* file comment */
    }
    sdate = SDate();
    stime = STime();

    for (i = 0; i < (int)strlen(sdate); i++)          /* no spaces allowed! */
        if (sdate[i] == ' ')  sdate[i] = '0';
    for (i = 0; i < (int)strlen(stime); i++)
        if (stime[i] == ' ')  stime[i] = '0';

    for (pdx = inf->pIdx; pdx != NULL; pdx = pdx->next)
    {
        if (pdx->flag & BIN_SELECTED)
        {
            sp = inf->cProject > 0 ? inf->asProject[pdx->iProject] : sdefproject;
            if (*sp == '\0')  sp = sdefproject;

            for (i = 0; i < (int)strlen(sp); i++)     /* no spaces allowed! */
                if (sp[i] == ' ')  sp[i] = '_';

            fprintf(fo, "%c%-12s %4d %2d %4d %2d %2d %ld %12s %s %s\n",
                (pdx->flag & BIN_DELETED) ? '-' : '>',
                pdx->sX, pdx->sam.year1, pdx->sam.period1,
                pdx->sam.year2, pdx->sam.period2, pdx->iFreq,
                pdx->lAddress, sp, sdate, stime);

            writeComment(fo, pdx->sComment);
        }
    }

return FCheckCloseW(fo);
}
/*-------------- END FReadInfFile/writeComment/FSaveInfFile ----------------*/

/*---------------------- FReadDataFile/FSaveDataFile -----------------------
 *  []Summary
 *    #include "in7bn7io.h"
 *
 *    bool FReadDataFile(struct InfType *inf)
 *    bool FSaveDataFile(struct InfType *inf, bool fAppend)
 *       inf       in: information structure
 *       fAppend   in: TRUE: append to file
 *
 *  []Return value
 *    TRUE if successful, FALSE otherwise.
 *
 *  []Description
 *    FReadDataFile reads the data file, using the description in the inf
 *        structure.
 *    FSaveDataFile saves the data file, using the description in the inf
 *        structure.
 *--------------------------------------------------------------------------*/
bool FReadDataFile(struct InfType *inf)
{
    int i, j, cx, ct, t1, t2, cxread, idata;
    FILE *fi;
    struct IdxType *pdx;

    if (!inf->cX)
    {
        PopErrorMessage("No variables selected");
        return FALSE;
    }
    if (!*SFileExt(inf->sDatafile))
        strcat(inf->sDatafile, (inf->iType == RS_BINARY_VAR) ? ".BN7" : ".DAT");

    if ( (fi = fopen(inf->sDatafile, "rb")) == NULL)
    {
        sprintf(sChart, sErrOpenFmt, inf->sDatafile);
        PopErrorMessage(sChart);
        return FALSE;
    }

    if ( (cx = inf->cX) > g_db.mxX)
    {
        cx = g_db.mxX;
        sprintf(sChart,"Can only load %d variables", cx);
        PopWarningMessage(sChart);
    }
    if ( (ct = inf->cT) > g_db.cT)
    {
        ct = g_db.cT;
        sprintf(sChart,"Can only load %d observations", ct);
        PopWarningMessage(sChart);
    }

    idata = -1;
    for (i = 0; i < cx; i++)                       /* failed to allocate cx */
    {                                             /* allocate one at a time */
        if ( (j = IDbAdd(1)) == -1)
            break;
        if (i == 0) idata = j;
    }
    if (cx != i)
    {
        sprintf(sChart,"Only %d variables loaded", i);
        PopWarningMessage(sChart);
    }
    cx = i;

    if (idata == -1)
        return FALSE;

    if (inf->iType == RS_BINARY_VAR)                      /* load selection */
    {
        for (i = 0, pdx = inf->pIdx; i < cx && pdx != NULL; pdx = pdx->next)
        {
            while (pdx != NULL && (pdx->flag & BIN_SELECTED) == 0)
                pdx = pdx->next;

            t1 = IDateObs(pdx->sam.year1, pdx->sam.period1);
            t2 = IDateObs(pdx->sam.year2, pdx->sam.period2);

            if (!FReadBinary(fi, pdx->lAddress, &g_db.mXt[idata + i][0],
                 t1, t2, g_db.cT, pdx->sX))
                IDbDel(idata + i);
            else
            {   strncpy(g_db.asX[idata + i], pdx->sX, NAME_LEN);
                g_db.asX[idata + i][NAME_LEN] = '\0';
                g_db.asComment[idata + i] = pdx->sComment;
                pdx->sComment = NULL;
                i++;
            }
        }
        cxread = i;
    }
    else
        PopErrorMessage("This code only reads .IN7/.BN7 files");

    fclose(fi);

    if (cxread != cx)
    {
        PopErrorMessage("Error reading data");
        return(FALSE);
    }

return TRUE;
}
bool FSaveDataFile(struct InfType *inf, bool fAppend)
{
    FILE *fo;  char openfmt[3]; int i, j, t1, t2;
    struct IdxType *pdx;

    if (!*SFileExt(inf->sDatafile))
        strcat(inf->sDatafile, (inf->iType == RS_BINARY_VAR) ? ".BN7" : ".DAT");

    openfmt[0] = (char)((fAppend) ? 'a' : 'w');
    openfmt[1] = (char)((inf->iType == RS_BINARY_VAR) ? 'b' : '\0');
    openfmt[2] = '\0';

    if ( (fo = fopen(inf->sDatafile, openfmt)) == NULL)
    {
        sprintf(sChart, sErrOpenFmt, inf->sDatafile);
        PopErrorMessage(sChart);
        return(FALSE);
    }

    if (inf->iType == RS_BINARY_VAR)
    {
        for (i = j = 0, pdx = inf->pIdx; j < inf->cX && pdx != NULL; pdx = pdx->next, i++)
        {
            if (!(pdx->flag & BIN_SELECTED))
                continue;
            t1 = IDateObs(pdx->sam.year1, pdx->sam.period1);
            t2 = IDateObs(pdx->sam.year2, pdx->sam.period2);

            pdx->lAddress = LSaveBinary(fo, &g_db.mXt[i][t1], t2 - t1 + 1,
                pdx->sam.year1, pdx->sam.period1, pdx->iFreq, pdx->sX);
            if (pdx->lAddress <= 0)
                pdx->flag &= ~BIN_SELECTED;
            j++;
        }
    }
    else
        PopErrorMessage("This code only saves .IN7/.BN7 files");

return FCheckCloseW(fo);
}
/*-------------------- END FReadDataFile/FSaveDataFile ---------------------*/


/*=========================== input/output =================================*/

/*----------------------- QuickLoadData/QuickSaveData ----------------------
 *  []Summary
 *    #include "in7bn7io.h"
 *
 *    void QuickLoadData(char *sFilename, char *sProject)
 *       sFilename in: .IN7 file to load from (if .IN7 is missing, it is
 *                     automatically appended). The name of the .BN7 file
 *                     is inferred from the IN7 file, and it must be in
 *                     the same directory.
 *       sProject  in: name of project to load or NULL in which case the
 *                     whole file is loaded.
 *    void QuickSaveData(char *sFilename)
 *       sFilename in: .IN7 file to write (if .IN7 is missing, it is
 *                     automatically appended). The .BN7 file will get
 *                     the same base name. Existing files with identical
 *                     names will be overwritten without warning.
 *
 *  []No return value
 *--------------------------------------------------------------------------*/
void QuickLoadData(char *sFilename, char *sProject)
{
    struct InfType *inf;  int i;

    if ( (inf = InfAlloc(sFilename)) == NULL)    /* allocate Info structure */
        return;

    DbKill();                                     /* kill current data base */

    inf->iType = RS_BINARY_VAR;
    if (!FReadInfFile(inf, TRUE, FALSE))        /* read info from .IN7 file */
        goto LReturn;

    if (sProject && *sProject)
    {   for (i = 0; i < inf->cProject; i++)
            if (strcmp(inf->asProject[i], sProject) == 0)
            {   SelectProject(inf, i);
                break;
            }
    }
    else
        SelectProject(inf, -1);

    InfSample(inf);                             /* set sample and frequency */
    InfToDb(inf);                         /* copy relevant info to database */

    FReadDataFile(inf);                                /* now read the data */

LReturn:

    InfFree(inf);
}
void QuickSaveData(char *sFilename)
{
    struct InfType *inf;

    if ( (inf = InfAlloc(sFilename)) == NULL)    /* allocate Info structure */
        return;

    inf->iType = RS_BINARY_VAR;
    strcpy(inf->sDatafile, inf->sFilename);           /* create binfilename */
    strcpy(SFileExt(inf->sDatafile), ".BN7");

    DbToInf(inf);                       /* copy relevant info from database */

    if (FSaveDataFile(inf, 0))                    /* save binary data first */
    {
        FSaveInfFile(inf, 0);                       /* now save information */
        g_db.fChange = FALSE;
    }

    InfFree(inf);
}
/*--------------------- END QuickLoadData/QuickSaveData --------------------*/



