// derived from program sv_mcl_est.ox (see http://staff.feweb.vu.nl/koopman/sv/)
#include <oxstd.h>
#include <oxfloat.h>
#include <oxdraw.h>
#include <oxprob.h>
#import <packages/oxmpi/loop>

#import <modelbase>

#include <packages/ssfpack/ssfpack.h>
#include <packages/ssfpack/ssfnong.ox>


class SvSim : Modelbase
{
	SvSim();

	decl m_iDistr;			 // distribution
	decl m_iSeed;			 // random number generator seed      
	decl m_iM;				 // number of simulation samples      
	decl m_iAnti;			 // number of antithetics: 1 or 2 or 4
	decl m_cAR, m_cMA;		 // ARMA(p,q)
	decl m_fShowProgress;	 // TRUE: use * to show progress
	decl m_iMeas, m_iLevel;
	decl m_dScale;
	decl m_mPhi, m_mOmega, m_mSigma, m_mJ_Phi, m_mJ_Omega, m_mIndex;
	decl m_mGamma, m_vTheta;

	SetSim(const iAnti, const iM);
	SetSeed(const iSeed);
	ARMA(const cAR, const cMA);
	SetDistribution(const iDist);

	SsfApproxModel(const iD);
	SsfMCLik();
	TransformPar(const vP, const pdAr1, const pdSig1, const pdSig2);
	Likelihood(const vP, const pdLik, const pvSco, const pmHes);
	StdErrPar(const vP);
	DrawMean(const iPlot);

	// simulation loop
	decl m_vYhat, m_vEhat, m_mApprox, m_mWgt, m_mKf, m_dLhat;
	draw(const iCtr);
	
	virtual GetPackageName();
	virtual GetPackageVersion();
	virtual GetParNames();
	virtual InitPar();
	SetStartPar(const vParFree);
	virtual DoEstimation(vPar);
	virtual Covar();
	virtual Output();
}
SvSim::SvSim()
{
	Modelbase();
	m_mJ_Phi = m_mJ_Omega = m_mIndex = <>;
	m_fShowProgress = FALSE;
	m_cAR = m_cMA = 0;
	m_iDistr = D_SV;
	m_iSeed = 5; 	
	m_iM = 250; 	
	m_iAnti = 4; 	
}
SvSim::GetPackageName()
{
	return "SV-sim";
}
SvSim::GetPackageVersion()
{
	return "0.01";
}
SvSim::GetParNames()
{
	decl i, asp = {};

	// ARMA names
	for (i = 0; i < m_cAR; ++i)
        asp ~= sprint("AR-", "%-2d", i + 1);
    for (i = 0; i < m_cMA; ++i)
        asp ~= sprint("MA-", "%-2d", i + 1);
		
	return asp ~ Modelbase::GetParNames() ~ "sigma1" ~ "sigma2";
}

SvSim::ARMA(const cAR, const cMA)
{
	m_cAR = cAR;
	m_cMA = cMA;
}
SvSim::SetDistribution(const iDist)
{
	m_iDistr = iDist;
}
SvSim::SetSim(const iAnti, const iM)
{
	m_iAnti = iAnti;
	m_iM = iM;
}
SvSim::SetSeed(const iSeed)
{
	m_iSeed = iSeed;
}

SvSim::SsfApproxModel(const iD)
{
	decl i, dconv;
	decl ms, mret, mtemp, mynz;
	
	mynz = (m_mY .== 0 .? 0.1 .: m_mY)';// to avoid log(0), but seems quite ad hoc??
	// approximation for non-Gaussian measurement models
	mret = SsfNonG_Expansion(iD, mynz, m_vTheta, m_dScale);
	for (i = 0; i < 50; i++)
	{
		ms = SsfCondDens(DS_SMO, mret[0][], m_mPhi, m_mOmega, m_mSigma, <>,
			m_mJ_Phi, m_mIndex, <>, m_mX' | mret);
		m_vTheta = mret[0][] - ms[m_iMeas][];

		mtemp = SsfNonG_Expansion(iD, mynz, m_vTheta, m_dScale);
 		dconv = max(fabs((mtemp - mret) ./ mret));
		
		mret = mtemp;
 
		if (m_fShowProgress) print("*");
		if (dconv < 1e-7) break;
	}

	if (m_fShowProgress) println("");
	return mret;
}

SvSim::SsfMCLik()
{
	decl i, mn, vr, d, dg, ms;

	m_mApprox = SsfApproxModel(m_iDistr);
	m_vYhat = m_mApprox[0][];
	
	SsfLik(&dg, &d, m_vYhat, m_mPhi, m_mOmega, m_mSigma,
		<>, m_mJ_Phi, m_mIndex, <>, m_mX' | m_mApprox);

	ms = SsfCondDens(DS_SMO, m_vYhat, m_mPhi, m_mOmega, m_mSigma,
		<>, m_mJ_Phi, m_mIndex, <>, m_mX' | m_mApprox);
	m_vEhat = ms[m_iMeas][];
	m_dLhat = SsfNonG_LogDensity(m_iDistr, m_mY', m_vYhat - m_vEhat, m_dScale)
		- SsfNonG_LogDensity(D_NORMAL, m_mY', m_vEhat, m_mApprox[1][]);

	if (m_iM == 0) return double(dg + m_dLhat);

	m_mKf  = KalmanFil(m_vYhat, m_mPhi, m_mOmega, m_mSigma,
		<>, m_mJ_Phi, m_mIndex, <>, m_mX' | m_mApprox);
	m_mWgt = SimSmoWgt(m_mGamma, m_mKf, m_mPhi, m_mOmega, m_mSigma,
		<>, m_mJ_Phi, m_mIndex, <>, m_mX' | m_mApprox);

	ranseed(m_iSeed);
	decl mweight = Loop::Run(draw, m_iM);

	mweight = meanc(mweight);
	mn = meanr(mweight);
	vr =  varr(mweight);
	return double(dg + m_dLhat + log(mn) + (0.5 * vr / (m_iM * mn^2)));
}
SvSim::draw(const iCtr)
{
	decl u = rann(1, columns(m_mApprox));
	decl ms = SimSmoDraw(m_mGamma, u, m_mWgt, m_mKf, m_mPhi, m_mOmega,
		m_mSigma, <>, m_mJ_Phi, m_mIndex, <>, m_mX' | m_mApprox);
	decl err = SsfNonG_Antithetic(m_iAnti, ms[m_iMeas][1:], m_vEhat, sumsqrr(u));
	return
		exp(SsfNonG_LogDensity(m_iDistr, m_mY', m_vYhat - err, m_dScale) -
		    SsfNonG_LogDensity(D_NORMAL, m_mY', err, m_mApprox[1][]) -
			m_dLhat);
}

SvSim::InitPar()
{
	if (!Modelbase::InitPar())
		return FALSE;

	// starting value procedure needs to be improved
	::GetSsfArma(zeros(1, m_cAR),
		zeros(1, m_cMA), 1, &m_mPhi, &m_mOmega, &m_mSigma);

	m_iLevel = 0;
	m_iMeas = columns(m_mPhi);
	if (m_mJ_Omega == <>)
		m_mIndex = constant(-1, m_mOmega);
	else
	{
		m_mIndex = m_mJ_Omega;
		if (m_mIndex[m_iMeas][m_iMeas] != -1)
		{
			println("*** Error: wrong model");
			return FALSE;
		}
	}

	// simulation administration
	m_mIndex[m_iMeas][m_iMeas] = max(-1, m_mJ_Phi, m_mJ_Omega) + 2;
	m_mGamma = zeros(m_mOmega);
	m_mGamma[m_iMeas][m_iMeas] = 1;
	m_vTheta = <>;

	// parameters of standard SV model
	SetParCount(m_cAR + m_cMA + 2);

	return TRUE;
}
SvSim::SetStartPar(const vParFree)
{
    if (m_iModelStatus < MS_PARAMS)
    	InitPar();					 // still need to initialize
	else
    	Modelbase::InitPar();		 // only re-initialize for existing model

	SetFreePar(vParFree);

	if (m_fPrint)
	{	println("Set starting values to:", GetFreePar()');
		decl d;
    	Likelihood(m_vPar, &d, 0, 0);
		println("initial log-likelihood: ", d);
	}
    
    return m_iModelStatus == MS_PARAMS;
}

SvSim::Likelihood(const vP, const pdLik, const pvSco, const pmHes)
{
    decl mapprox, dar, dsig;

	// model parameters
	TransformPar(vP, &dar, &dsig, &m_dScale);
	::GetSsfArma(dar, <>, dsig, &m_mPhi, &m_mOmega, &m_mSigma);
	
	pdLik[0] = SsfMCLik() / m_cT;
	return 1;   
}

SvSim::DoEstimation(vPar)
{
    // maximum likelihood estimation 
    m_iResult = MaxBFGS(Likelihood, &vPar, &m_dLogLik, 0, TRUE);

	if (m_iResult == MAX_CONV || m_iResult == MAX_WEAK_CONV)
		Covar();

	return {vPar, "BFGS", TRUE};
}

SvSim::Covar()
{
	decl covar, result, vp = GetFreePar(), vpsave = vp, d;

	m_mCovar = <>;

    result = Num2Derivative(Likelihood, vp, &covar);
	// reset original pars and call Likelihood, to avoid side-effects
	SetFreePar(vpsave);
	Likelihood(vpsave, &d, 0, 0);
    if (!result)
    	print("Covar() failed in numerical second derivatives\n");
	else
	    m_mCovar = invertgen(-covar, 30);
}

SvSim::DrawMean(const iPlot)
{
	decl i, vyhat;
    decl mapprox, ms, mshat, mw, mcum;
	decl a, dlhat, old;
	
	old = m_mOmega[0][0];

	mapprox = SsfApproxModel(m_iDistr);
	vyhat = mapprox[0][];

	mshat = SsfCondDens(DS_SMO, vyhat, m_mPhi, m_mOmega, m_mSigma,
		<>, m_mJ_Phi, m_mIndex, <>, m_mX' | mapprox);
	dlhat =	SsfNonG_LogDensity(m_iDistr, m_mY', vyhat - mshat[m_iMeas][], m_dScale) -
			SsfNonG_LogDensity(D_NORMAL, m_mY', mshat[m_iMeas][], mapprox[1][]);
	
	mw = zeros(1, m_iM);
	mcum = zeros(2, m_cT);
	ranseed(m_iSeed);
	for (i=0; i<m_iM; i++)
	{
		// p(y|theta) / g(y|theta)
		ms = SsfCondDens(ST_SIM, vyhat, m_mPhi, m_mOmega, m_mSigma,
			<>, m_mJ_Phi, m_mIndex, <>, m_mX' | mapprox);
		mw[0][i] = exp(SsfNonG_LogDensity(m_iDistr, m_mY', ms[m_iMeas][], m_dScale) -
					   SsfNonG_LogDensity(D_NORMAL, m_mY', vyhat - ms[m_iMeas][], mapprox[1][]) -
					   dlhat);
		a = ms[m_iLevel][];
		mcum[0][] += mw[0][i] * a;
		mcum[1][] += mw[0][i] * (a .* a);
		if (m_fShowProgress && imod(i, 100) == 0) print("+");
	}
	if (m_fShowProgress) println("");
	a = sumr(mw[0][]);

	mcum[0][] /= a;
	mcum[1][] /= a;
	mcum[1][] -= mcum[0][] .* mcum[0][];

	m_mOmega[0][0] = old;

	decl mn  = sqrt(m_dScale*exp(mcum[0][]));
	decl se1 = sqrt(m_dScale*exp(mcum[0][] - 2.0 * sqrt(mcum[1][])));
	decl se2 = sqrt(m_dScale*exp(mcum[0][] + 2.0 * sqrt(mcum[1][])));
	DrawTitle(iPlot, "Conditional mean estimation");
	DrawTMatrix(iPlot, fabs(m_mY[][0]') | mn | se1 | se2, {"Abs Diff exchange data", "ar1"}, 1, 1, 1, 0);
	DrawTMatrix(iPlot+1,
		mcum[0][] |
		mcum[0][] - 2.0 * sqrt(mcum[1][]) |
		mcum[0][] + 2.0 * sqrt(mcum[1][]), {"ar1"}, 1, 1, 1, 0);
}
SvSim::Output()
{
	Modelbase::Output();
	println("Phi    ", m_mPhi);
	println("Omega  ", m_mOmega);
	println("Sigma2 ", m_dScale);
}

SvSim::TransformPar(const vP, const pdAr1, const pdSig1, const pdSig2)
{
	pdAr1[0]  = exp(vP[0]) / (1.0 + exp(vP[0]));
	pdSig1[0] = exp(vP[1]);
	pdSig2[0] = exp(2.0 * vP[2]);
}


main()
{
	// data
	decl mdata = loadmat("svpdx.dat");
	decl sv = new SvSim();
	sv.Create(1, 1, 1, rows(mdata), 1);
	sv.Append(mdata[][0], "pdx");
	sv.Select(Modelbase::Y_VAR, {"pdx", 0, 0});
	sv.SetSelSample(-1, 1, -1, 1);

	// model
	sv.ARMA(1, 0);
	sv.SetDistribution(D_SV);
	// model administration

	sv.SetStartPar(<5.0;-2.0;-0.5>);

    MaxControl(-1, 5, 1);
	MaxControlEps(1e-4, 1e-4);			// tighter convergence criteria

	decl time = Loop::Timer();
	sv.Estimate(); // maximum likelihood estimation
	println("time = ", Loop::Timer() - time, " secs.");	       

//	sv.DrawMean(0); // graphs mean
	ShowDrawWindow();
}
